%This copy distributed with jovcode.zip, which contains only those
%Palamedes files that are needed to replicate results presented in:
%
%Prins, N. (2012). The psychometric function: The lapse rate revisited.
%Journal Of Vision.
%
%
%In order to enjoy the full functionality of the Palamedes Toolbox 
%(including help comments for each user-end routine), visit 
%www.palamedestoolbox.org and download the full version of the toolbox.
%
%This code may not be reproduced and/or distributed in original or
%   modified form under a different name and/or authorship.
%
%Palamedes toolbox citation:
%
%Prins, N. & Kingdom, F.A.A. (2009) Palamedes: Matlab routines for
%analyzing psychophysical data. www.palamedestoolbox.org.
%
% Introduced: Palamedes version 1.0.0 (NP)
% Modified: Palamedes version 1.0.2, 1.2.0, 1.3.0, 1.3.1, 1.4.0 
%   (see History.m)

function [paramsValues LL exitflag output] = PAL_PFML_Fit(StimLevels, NumPos, OutOfNum, searchGrid, paramsFree, PF, varargin)

options = [];
lapseLimits = [];
guessLimits = [];
lapseFit = 'default';
gammaEQlambda = logical(false);

if ~isempty(varargin)
    NumOpts = length(varargin);
    for n = 1:2:NumOpts
        valid = 0;
        if strncmpi(varargin{n}, 'SearchOptions',7)
            options = varargin{n+1};
            valid = 1;
        end
        if strncmpi(varargin{n}, 'lapseLimits',6)
            if paramsFree(4) == 0 && ~isempty(varargin{n+1})
                message = ['Lapse rate is not a free parameter: ''lapseLimits'' argument ignored'];
                warning('PALAMEDES:PFML_Fit_lapseLimits',message)
            else
                lapseLimits = varargin{n+1};
            end
            valid = 1;
        end
        if strncmpi(varargin{n}, 'guessLimits',6)
            if paramsFree(3) == 0 && ~isempty(varargin{n+1})
                message = ['Guess rate is not a free parameter: ''guessLimits'' argument ignored'];
                warning('PALAMEDES:PFML_Fit_guessLimits',message)
            else
                guessLimits = varargin{n+1};
            end
            valid = 1;
        end
        if strncmpi(varargin{n}, 'lapseFit',6)
            if paramsFree(4) == 0 && ~strncmpi(varargin{n+1},'def',3)
                message = ['Lapse rate is not a free parameter: ''lapseFit'' argument ignored'];
                warning(message);
            else
                lapseFit = varargin{n+1};
            end
            valid = 1;
        end
        if strncmpi(varargin{n}, 'gammaEQlambda',6)
            gammaEQlambda = logical(varargin{n+1});
            valid = 1;
        end
        if valid == 0
            message = [varargin{n} ' is not a valid option. Ignored.'];
            warning(message);
        end        
    end            
end

if ~isempty(guessLimits) && gammaEQlambda
    message = ['Guess rate is constrained to equal lapse rate: ''guessLimits'' argument ignored'];
    warning('PALAMEDES:PFML_Fit_guessLimits',message)
    guessLimits = [];
end

[StimLevels NumPos OutOfNum] = PAL_PFML_GroupTrialsbyX(StimLevels, NumPos, OutOfNum);

if isstruct(searchGrid)
    if gammaEQlambda
        searchGrid.gamma = 0;
    end
    searchGrid = PAL_PFML_BruteForceFit(StimLevels, NumPos, OutOfNum, searchGrid, PF, 'lapseFit',lapseFit,'gammaEQlambda',gammaEQlambda);
end

if gammaEQlambda
    searchGrid(3) = searchGrid(4);
    paramsFree(3) = 0;
end

paramsFreeVals = searchGrid(paramsFree == 1);
paramsFixedVals = searchGrid(paramsFree == 0);

if isempty(paramsFreeVals)
    negLL = PAL_PFML_negLL(paramsFreeVals, paramsFixedVals, paramsFree, StimLevels, NumPos, OutOfNum, PF,'lapseFit',lapseFit);
    exitflag = 1;
    output = [];
else
    switch lower(lapseFit(1:3))
        case 'iap'
            len = length(NumPos);        
            if ~gammaEQlambda
                lambda = 1 - NumPos(len)./OutOfNum(len);
                if ~isempty(lapseLimits)
                    lambda = min(lambda,lapseLimits(2));
                    lambda = max(lambda,lapseLimits(1));
                end
                paramsFixedVals(length(paramsFixedVals)+1) = lambda; %lapse rate
                paramsFreeVals = paramsFreeVals(1:length(paramsFreeVals)-1);    %set lapse rate estimate as fixed value
                paramsFree(4) = 0;               
                [paramsFreeVals negLL exitflag output] = PAL_minimize(@PAL_PFML_negLL, paramsFreeVals, options, paramsFixedVals, paramsFree, StimLevels(1:len-1), NumPos(1:len-1), OutOfNum(1:len-1), PF,'gammaEQlambda',gammaEQlambda,'guessLimits',guessLimits);        
                negLL = negLL - log((1 - lambda).^NumPos(len)) - log(lambda.^(OutOfNum(len)-NumPos(len)));
            else
                lambda = 1 - (NumPos(len)+(OutOfNum(1)-NumPos(1)))./(OutOfNum(len)+OutOfNum(1));                
                if ~isempty(lapseLimits)
                    lambda = min(lambda,lapseLimits(2));
                    lambda = max(lambda,lapseLimits(1));
                end
                paramsFixedVals(length(paramsFixedVals)+1) = lambda;
                paramsFixedVals(length(paramsFixedVals)) = lambda;
                paramsFreeVals = paramsFreeVals(1:length(paramsFreeVals)-1);
                paramsFree(4) = 0;
                [paramsFreeVals negLL exitflag output] = PAL_minimize(@PAL_PFML_negLL, paramsFreeVals, options, paramsFixedVals, paramsFree, StimLevels(2:len-1), NumPos(2:len-1), OutOfNum(2:len-1), PF,'gammaEQlambda',gammaEQlambda);        
                negLL = negLL - log((1 - lambda).^(NumPos(len)+(OutOfNum(1)-NumPos(1)))) - log(lambda.^(OutOfNum(len)-NumPos(len)+NumPos(1)));
            end            
        case {'nap', 'jap', 'def'}
            [paramsFreeVals negLL exitflag output] = PAL_minimize(@PAL_PFML_negLL, paramsFreeVals, options, paramsFixedVals, paramsFree, StimLevels, NumPos, OutOfNum, PF,'lapseLimits',lapseLimits,'guessLimits',guessLimits,'lapseFit',lapseFit,'gammaEQlambda',gammaEQlambda);
    end
end

paramsValues = zeros(1,4);
paramsValues(paramsFree == 1) = paramsFreeVals;
paramsValues(paramsFree == 0) = paramsFixedVals;

if gammaEQlambda
    paramsValues(3) = paramsValues(4);
end

LL = -1*negLL;